<?php


namespace App\Models\Orders\Service;


use App\Exceptions\WalletException;
use App\Models\Carts\Cart;
use App\Models\Commons\Donation;
use App\Models\Delivery\Delivery;
use App\Models\Orders\Order;
use App\Models\Orders\OrderProduct;
use App\Models\Orders\OrderVariation;
use App\Models\Packages\Basket;
use App\Models\Packages\Package;
use App\Services\TransactionService;
use Exception;
use Illuminate\Http\Request;

class OrderCreateService
{
    protected TransactionService $transaction;

    public function __construct(TransactionService $transactionService)
    {
        $this->transaction = $transactionService;
    }

    /**
     * @throws Exception
     */
    public function create(Request $request): ?Order
    {
        $id = auth()->id();
        $type = $request->type;

        $orderId = 'INAR' . floor(time() - 999999999);
        $paymentMethod = $request->payment_method;
        $paymentId = $request->payment_id;
        $paymentOrderId = $request->payment_order_id;

        $data = [
            'order_id' => $orderId,

            'name' => '',

            'payment_method' => $paymentMethod,
            'payment_id' => $paymentId,
            'payment_order_id' => $paymentOrderId,
            'payment_status' => Order::PAYMENT_PENDING,

            'type' => $type,

            'customer_id' => $id,
            'address_id' => $request->address_id,
            'women_delivery' => filter_var($request->women_delivery, FILTER_VALIDATE_BOOLEAN),
        ];

        /** @var Order $order */
        $order = Order::query()->create($data);

        $data = [];

        switch ($type) {
            case "cart":
                $data = $this->cartOrder($order);
                break;
            case "package":
                $data = $this->packageOrder($request, $order);
                break;
            case "basket":
                $data = $this->basketOrder($request);
                break;
            case "wallet":
                $data = $this->walletOrder($request);
                break;
            case "donation":
                $data = $this->donationOrder($request);
                break;
            case "debt":
                $data = $this->debtOrder($request);
                break;
            default:
                echo "Your favorite color is neither red, blue, nor green!";
        }

        $order->update($data);

        if ($paymentMethod == 'wallet' && $type != "wallet") {
            try {
                $id = $this->transaction->debit($id, $data['total']);

                if($order->type == Order::TYPE_DEBT) {
                    $this->transaction->undebt($order->customer_id, $order->total['amount']);
                }

                $data['payment_id'] = $id;
                $data['payment_status'] = Order::PAYMENT_SUCCESS;
                $data['status'] = Order::ORDER_CONFIRMED;

                $order->update($data);

            } catch (WalletException $e) {
                $data['payment_id'] = '';
                $data['payment_status'] = Order::PAYMENT_FAILED;
                $data['status'] = Order::ORDER_FAILED;
                $order->update($data);

                throw new Exception($e->getMessage());
            }
        }

        if ($paymentMethod == 'coupon' && $type != "wallet") {
            try {
                $id = $this->transaction->debitCoupon($id, $data['total']);

                $data['payment_id'] = $id;
                $data['payment_status'] = Order::PAYMENT_SUCCESS;
                $data['status'] = Order::ORDER_CONFIRMED;

                $order->update($data);

            } catch (WalletException $e) {
                $data['payment_id'] = '';
                $data['payment_status'] = Order::PAYMENT_FAILED;
                $data['status'] = Order::ORDER_FAILED;
                $order->update($data);

                throw new Exception($e->getMessage());
            }
        }

        return $order->fresh();
    }

    protected function cartOrder(Order $order): array
    {
        $orderId = $order->id;
        $carts = Cart::query()
            ->where('customer_id', auth()->id())
            ->with('product', 'variant')
            ->get();
        $firstProduct = $carts->first()->product;

        $mrp = 0;
        $total = 0;

        foreach ($carts as $cart) {
            $mrp += $cart->mrp['amount'];
            $total += $cart->price['amount'];

            $orderProduct = OrderProduct::query()->create([
                'order_id' => $orderId,

                'product_id' => $cart->product_id,
                'total_unit' => $cart->variant->unit,

                'price' => $cart->price['amount'],
            ]);

            $orderVariation = OrderVariation::query()->create([
                'order_product_id' => $orderProduct->id,

                'variant_id' => $cart->variant_id,
                'quantity' => $cart->quantity,
                'total' => $cart->price['amount'],

                'remaining_quantity' => 0,
            ]);

            Delivery::query()->create([
                'order_id' => $orderId,

                'postcode' => $order->address->pin,
                'area' => $order->address->road,

                'customer_id' => auth()->id(),

                'product_id' => $orderProduct->id,

                'date' => now(),
                'time' => '',

                'variation_id' => $orderVariation->id,
                'quantity' => $orderVariation->quantity,

                'status' => 'confirmed',
            ]);
        }

        return [
            'name' => $firstProduct->name,
            'image' => $firstProduct->image,
            'mrp' => $mrp,
            'total' => $total,
            'discount' => $mrp - $total,
        ];
    }

    protected function basketOrder(Request $request): array
    {
        $basket = Basket::query()->find($request->basket_id);
        $total = $request->amount;

        return [
            'name' => $basket->name,
            'image' => $basket->image,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'basket_id' => $basket->id,
            'mrp' => $total,
            'total' => $total,
            'discount' => 0,
        ];
    }

    protected function packageOrder(Request $request, Order $order): array
    {
        $orderId = $order->id;
        $package = Package::where('id', $request->package_id)->with('products')->first();

        foreach ($package->products as $product) {

            $orderProduct = OrderProduct::query()->create([
                'order_id' => $orderId,

                'product_id' => $product->product_id,
                'price' => $product->price,
                'total_unit' => $product->total_unit,
            ]);

            foreach ($product->variations as $productVariant) {
                OrderVariation::create([
                    'order_product_id' => $orderProduct->id,

                    'variant_id' => $productVariant->variant_id,

                    'quantity' => $productVariant->quantity,
                    'remaining_quantity' => $productVariant->quantity,

                    'total' => $productVariant->total,
                ]);
            }
        }

        return [
            'name' => $package->name,
            'image' => $package->image,
            'package_id' => $package->id,
            'mrp' => $package->mrp,
            'total' => $package->price,
            'discount' => $package->discount,
        ];
    }

    protected function walletOrder(Request $request): array
    {
        $amount = $request->amount;

        return [
            'name' => 'Wallet Recharge',

            'mrp' => $amount,
            'total' => $amount,
            'discount' => 0,
        ];
    }

    protected function donationOrder(Request $request): array
    {
        $amount = $request->amount;
        $donation = Donation::query()->find($request->donation_id);

        return [
            'name' => $donation->title,
            'image' => $donation->image,

            'donation_id' => $donation->id,

            'mrp' => $amount,
            'total' => $amount,
            'discount' => 0,
        ];
    }

    protected function debtOrder(Request $request): array
    {
        $amount = $request->amount;

        return [
            'name' => 'Due Payment',

            'mrp' => $amount,
            'total' => $amount,
            'discount' => 0,
        ];
    }
}
