<?php

namespace App\Models\Orders;

use App\Casts\Money;
use App\Models\Addresses\Address;
use App\Models\Delivery\Delivery;
use App\Models\Packages\Package;
use App\Models\User;
use App\Shared\Active;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Znck\Eloquent\Traits\BelongsToThrough as Znck;

/***
 * @property int $id
 * @property string $order_id
 */
class Order extends Model
{
    use HasFactory, Active, Znck;

    const PAYMENT_PENDING = 'pending';
    const PAYMENT_FAILED = 'failed';
    const PAYMENT_SUCCESS = 'paid';
    const PAYMENT_DUE = 'due';

    const PM_COD = 'cod';
    const PM_WALLET = 'wallet';
    const PM_COUPON = 'coupon';
    const PM_BANK = 'netbanking';

    const ORDER_PENDING = 'pending';
    const ORDER_CONFIRMED = 'confirmed';
    const ORDER_FAILED = 'cancelled';
    const ORDER_ACTIVE = 'active';
    const ORDER_COMPLETED = 'completed';

    const TYPE_CART = 'cart';
    const TYPE_PACKAGE = 'package';
    const TYPE_BASKET = 'basket';
    const TYPE_WALLET = 'wallet';
    const TYPE_DONATION = 'donation';
    const TYPE_DEBT = 'debt';

    protected $fillable = [
        'order_id',

        'name',
        'image',

        'payment_method',
        'payment_status',
        'payment_id',
        'payment_order_id',

        'mrp',
        'discount',
        'total',

        'start_date',
        'end_date',

        'type',

        'address_id',
        'customer_id',

        'package_id',
        'basket_id',

        'vendor_id',

        'confirmed',

        'women_delivery',
        'donation_id',

        'verified',
        'status',
    ];

    protected $casts = [
        'verified' => 'boolean',
        'women_delivery' => 'boolean',
        'confirmed' => 'boolean',
        'mrp' => Money::class,
        'total' => Money::class,
    ];

    protected $appends = [
        'duration',
        'can_cancel',
        'show_confirm',
    ];

    public function customer(): BelongsTo
    {
        return $this
            ->belongsTo(User::class, 'customer_id');
    }

    public function address(): BelongsTo
    {
        return $this
            ->belongsTo(Address::class, 'address_id');
    }

    public function products(): HasMany
    {
        return $this
            ->hasMany(OrderProduct::class)
            ->with('product', 'variations.variant');
    }

    public function deliveries(): HasMany
    {
        return $this
            ->hasMany(Delivery::class, 'order_id');
    }

    public function package(): BelongsTo
    {
        return $this
            ->belongsTo(Package::class);
    }

    public function getDurationAttribute(): int
    {
        return $this->package->duration ?? 0;
    }

    public function getCanCancelAttribute(): bool
    {
        if ($this->type != self::TYPE_CART) return false;

        return !in_array($this->status, [self::ORDER_FAILED, self::ORDER_COMPLETED]);
    }

    public function getShowConfirmAttribute(): bool
    {
        if ($this->type != self::TYPE_CART) return false;

        return !$this->confirmed;
    }
}
