<?php

namespace App\Http\Livewire;

use App\Casts\Money;
use App\Models\Delivery\Delivery;
use App\Models\Orders\Order;
use DateTime;
use Illuminate\Database\Eloquent\Builder;
use Rappasoft\LaravelLivewireTables\DataTableComponent;
use Rappasoft\LaravelLivewireTables\Views\Column;

class DeliveriesTable extends DataTableComponent
{
    public Order $order;
    public DateTime $date;

    public array $bulkActions = [
        Delivery::STATUS_DELIVERED => 'DELIVERED',
        Delivery::STATUS_CANCELLED => 'CANCELLED',
    ];

    public function columns(): array
    {
        return [
            Column::make('CUSTOMER', 'customer.name')->format(function ($value) {
                return view('admin.shared.avatar')->with(['name' => $value]);
            })->searchable(),

            Column::make('PRODUCT', 'product.image')->format(function ($value, $column, $row) {
                return view('admin.shared.image')->with([
                    'image' => $value,
                    'name' => $row->product->name,
                ]);
            }),

            Column::make('QUANTITY', 'quantity')->format(function ($value, $column, $row) {
                return $row->quantity . ' X ' . ($row->variant->unit ?? '');
            }),

            Column::make('PRICE')->format(function ($value, $column, $row) {
                $price = $row->variant->price['amount'] ?? 0;
                $amount = Money::format($row->quantity * $price);

                return '<span class="text-green">' . $amount['currency'] . '</span>';
            })
                ->asHtml(),

            Column::make('CHOICE', 'order.women_delivery')->format(function ($value) {
                if ($value) return '<i class="fas fa-female" style="color: #601FCC"></i>';

                return '<i class="fas fa-male mr-1"></i><i class="fas fa-female"></i>';
            })->asHtml(),

            Column::make('DATE', 'date'),
            Column::make('TIME', 'time'),

            Column::make('ADDRESS')->format(function ($value, $column, $row) {
                return $row->address->road . ', ' . $row->address->pin;
            }),

            Column::make('STATUS', 'status')->format(function ($value, $column, $row) {
                return view('admin.deliveries.status', compact('row'));
            }),

            Column::make('ACTIONS')->format(function ($value, $column, $row) {
                return view('admin.deliveries.action', compact('row'));
            }),
        ];
    }

    public function query(): Builder
    {
        if (!isset($this->order)) {
            return Delivery::query()
                ->whereDate('date', $this->date)
                ->with(['customer', 'product', 'variant', 'order', 'address'])
                ->latest();
        }

        return Delivery::query()
            ->where('order_id', $this->order->id)
            ->with(['customer', 'product', 'variant', 'order', 'address'])
            ->latest();
    }

    public function cancelled(): void
    {
        if ($this->selectedRowsQuery->count() > 0) {
            Delivery::query()
                ->whereIn('id', $this->selectedKeys())
                ->update(['status' => 'cancelled']);
        }

        $this->selected = [];

        $this->resetBulk();
    }

    public function delivered(): void
    {
        if ($this->selectedRowsQuery->count() > 0) {
            Delivery::query()
                ->whereIn('id', $this->selectedKeys())
                ->update(['status' => 'delivered']);
        }

        $this->resetBulk();
    }
}
