<?php

namespace App\Http\Controllers\API\Delivery\Delivery;

use App\Http\Controllers\Controller;
use App\Http\Resources\Delivery\DeliveryOrderResource;
use App\Models\Delivery\Delivery;
use App\Models\Orders\Order;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Carbon;

class DeliveryController extends Controller
{
    public function index(): AnonymousResourceCollection
    {
        $orders = Order::query()
            ->has('address')
            ->has('customer')
            ->with('address')
            ->with('deliveries', function ($query) {
                return $query
                    ->where('delivery_id', auth()->id())
                    ->where('status', Delivery::STATUS_CONFIRMED)
                    ->has('product')
                    ->has('variation')
                    ->with(['product', 'variation']);
            })
            ->has('deliveries')
            ->latest()
            ->get()
            ->filter(function ($order, $key) {
                return count($order->deliveries) > 0;
            });

        return DeliveryOrderResource::collection($orders);
    }

    public function update(Request $request): AnonymousResourceCollection
    {
        $deliveries = json_decode($request->delivery_ids);
        $status = $request->status;


        $order = Delivery::query()
            ->find($deliveries[0])
            ->order;

        $type = $order->type;


        if ($status == Delivery::STATUS_DELIVERED) {
            if ($request->payment_method == Order::PM_COD) {
                $this->transaction->debit(auth()->id(), $request->amount);
            }


            if ($type == Order::TYPE_CART) {
                $order->update(['status' => Order::ORDER_COMPLETED]);
            }
        }

        if($status == Delivery::STATUS_CANCELLED && $type == Order::TYPE_CART) {
            $order->update(['status' => Order::ORDER_FAILED]);
        }


        Delivery::query()
            ->whereIn('id', $deliveries)
            ->update(['status' => $status]);

        return $this->index();
    }

    public function count(Request $request): int
    {
        $query = Delivery::query()
            ->where('delivery_id', auth()->id())
            ->where('status', Delivery::STATUS_DELIVERED);

        if (isset($request->start) && isset($request->end)) {
            $start = Carbon::parse($request->start);
            $end = Carbon::parse($request->end);

            return $query
                ->whereBetween('date', [$start, $end])
                ->count();
        }

        return $query->count();
    }
}
