<?php

namespace App\Http\Controllers\API\Customer\Order;

use App\Http\Controllers\Controller;
use App\Http\Resources\Delivery\DeliveryResource;
use App\Models\Delivery\Delivery;
use App\Models\Orders\Order;
use App\Models\Orders\OrderVariation;
use Carbon\Carbon;
use Illuminate\Http\Request;

class DeliveryController extends Controller
{
    public function index()
    {
        if (request('order_id') == null) {
            return Delivery::query()
                ->where('customer_id', auth()->id())
                ->with('item', 'variation')
                ->latest()
                ->get();
        }

        $deliveries = Delivery::query()
            ->where('order_id', request('order_id'))
            ->with(['product', 'variation'])
            ->latest()
            ->get();

        return DeliveryResource::collection($deliveries);
    }

    public function store(Request $request)
    {
        $data = $request->all();
        $data['customer_id'] = auth()->id();

        $order = Order::query()->where('id', $request->order_id)->first();

        $quantity = $data['quantity'];
        $variation = OrderVariation::query()->find($data['variation_id']);
        $data['status'] = Delivery::STATUS_PENDING;

        $data['postcode'] = $order->address->pin;
        $data['area'] = $order->address->road;

        Delivery::query()->create($data);

        $variation->update([
            'remaining_quantity' => $variation->remaining_quantity - $quantity,
        ]);
    }

    public function update(Request $request)
    {
        if (isset($request['time'])) {
            Delivery::query()
                ->where('order_id', $request->order_id)
                ->where('status', Delivery::STATUS_PENDING)
                ->whereDate('date', $request->date)
                ->update(['time' => $request->time]);
        } else {
            Delivery::query()
                ->where('order_id', $request->order_id)
                ->where('status', Delivery::STATUS_PENDING)
                ->update(['status' => Delivery::STATUS_CONFIRMED]);
        }
    }

    public function destroyDelivery(Delivery $delivery)
    {
        $variation = OrderVariation::query()->find($delivery->variation_id);

        $variation->update([
            'remaining_quantity' => $variation->remaining_quantity + $delivery->quantity,
        ]);


        $delivery->delete();
    }

    public function destroy()
    {
        $deliveries = Delivery::query()
            ->where('order_id', request('order_id'))
            ->whereIn('status', [Delivery::STATUS_CONFIRMED, Delivery::STATUS_PENDING, Delivery::STATUS_CANCELLED])
            ->where('date', Carbon::parse(request('date')))
            ->with('variation')
            ->get();

        foreach ($deliveries as $item) {
            $quantity = $item->quantity;

            $item->variation->update([
                'remaining_quantity' => $item->variation->remaining_quantity + $quantity,
            ]);

            $item->delete();
        }
    }
}
