<?php

namespace App\Http\Controllers\API\Customer\Cart;

use App\Casts\Money;
use App\Http\Resources\Cart\CartResource;
use App\Models\Carts\Cart;
use App\Models\Carts\Services\CartService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class CartController extends Controller
{
    public function index(): array
    {
        $carts = Cart::query()
            ->where('customer_id', auth()->id())
            ->with('product', 'variant')
            ->latest()
            ->get();

        $mrp = $carts->sum(function ($cart) {
            return $cart->mrp['amount'];
        });

        $price = $carts->sum(function ($cart) {
            return $cart->price['amount'];
        });

        $discount = $mrp - $price;

        return [
            'mrp' => Money::format($mrp),
            'price' => Money::format($price),
            'discount' => Money::format($discount),
            'carts' => CartResource::collection($carts),
        ];
    }

    public function store(Request $request, CartService $service): JsonResponse
    {
        $data = $request->all();

        $validator = Validator::make($data, [
            'quantity' => ['required'],
            'product_id' => ['required'],
            'variant_id' => ['required'],
        ]);

        if ($validator->fails()) {
            return response()
                ->json(['error' => $validator->errors()->first()], 422);
        }

        $data['customer_id'] = auth()->id();

        $cart = Cart::query()
            ->where([
                'customer_id' => auth()->id(),
                'product_id' => $data['product_id'],
                'variant_id' => $data['variant_id'],
            ])->first();

        if (isset($cart)) {
            $cart->increment('quantity', $data['quantity']);

            $service->updatePrice($cart->fresh());

            return response()->json(['status' => false]);
        }

        $cart = Cart::query()->create($data);
        $service->updatePrice($cart);

        return response()->json(['status' => true]);
    }

    public function delete(Cart $cart): array
    {
        $cart->delete();

        return $this->index();
    }

    public function cartCount(): int
    {
        return Cart::query()
            ->where('customer_id', auth()->id())
            ->count();
    }

    public function updateQuantity(Request $request, Cart $cart, CartService $service)
    {
        $data = $request->all();

        $validator = Validator::make($data, [
            'quantity' => ['required'],
        ]);

        if ($validator->fails()) {
            return response()
                ->json(['error' => $validator->errors()->first()], 422);
        }

        $cart->update($data);

        $service->updatePrice($cart->fresh());

        return $this->index();
    }
}
