<?php

namespace App\Http\Controllers\API\Auth;

use App\Http\Controllers\Controller;
use App\Http\Resources\Customer\CustomerResource;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class VendorAuthController extends Controller
{
    public function login(Request $request): JsonResponse
    {
        $data = $request->all();

        $validator = Validator::make($data, [
            'phone' => ['required', 'digits:10'],
            'password' => ['required', 'string'],
        ]);

        if ($validator->fails()) {
            return response()
                ->json(['error' => $validator->errors()->first()], 422);
        }

        $vendor = User::query()->role('merchant')->wherePhone($data['phone'])->first();

        if (!$vendor || !Hash::check($data['password'], $vendor->password)) {
            return response()->json([
                'status' => false,
                'message' => 'The provided credentials are incorrect.'
            ], 401);
        }

        return response()->json([
            'token' => $vendor->createToken($data['phone'], ['role:vendor'])->plainTextToken,
            'user' => CustomerResource::make($vendor),
        ]);
    }

    public function register(Request $request): JsonResponse
    {
        $data = $request->all();

        $validator = Validator::make($data, [
            'name' => ['required', 'string', 'max:255'],
            'phone' => ['required', 'digits:10', 'unique:users'],
            'email' => ['email', 'unique:users'],
            'type' => ['required'],
            'password' => ['required', 'string', 'min:6'],
        ]);

        if ($validator->fails()) {
            return response()
                ->json(['error' => $validator->errors()->first()], 422);
        }

        $data['password'] = Hash::make($data['password']);

        /** @var User $vendor */
        $vendor = User::query()->create($data);

        $vendor->assignRole('merchant');
        $vendor->wallet()->create();

        return response()->json([
            'token' => $vendor->createToken($data['phone'], ['role:vendor'])->plainTextToken,
            'user' => CustomerResource::make($vendor),
        ]);
    }

    public function logout()
    {
        auth()->user()->currentAccessToken()->delete();
    }
}
